*! version 1.0.0  1aug2014  dcs

* based on official Stata's -var_p- and -svar_p-

program define svarih_bac_p100, sortpreserve

version 11.2

syntax newvarname [if] [in] ,       ///  'varlist' from 'newvarname' is used as a stub when option hdecomp is used
                                    ///  e.g. if 'hdecomp hdshock(eq1)', then the HD for the shock in eq1 for all endogvars is calced,
                                    ///  the new varnames being 'stub'eqname
                                    ///  must parse using 'newvarname' as this automatically defines `typlist'
            [ xb                    ///  default
              Residuals             ///
              Shocks                ///
              HDecomp               ///
              HDShock(string)   ///
              HDBeg(string)         /// may be date or date number
              HDEnd(string)         /// may be date or date number; TODO: also allow to specify # of periods
              EQuation(string)      ///
              ]
    
    qui tsset
    local tvar   `r(timevar)'  // locals needed for option -hdecomp-
    local unit1  `r(unit1)'
    local tsfmt  `r(tsfmt)'

    marksample touse, novarlist

    if "`e(cmd)'" != "svarih" | "`e(method)'"!="Bacchiocchi" {
        di as err "'svarih_bac_p100' only works after -svarih bacciocchi-"
        exit 301
    }

    local nstats : word count `xb' `residuals' `shocks' `hdecomp'
    if `nstats' > 1 {
        di as err "more than one statistic specified"
        exit 198
    }

    if "`xb'`residuals'`shocks'`equation'"!="" & `"`hdecomp'`hdbeg'`hdend'`hdshock'"'!="" {
        di as err `"Invalid combination of options."'
        exit 198
    }

    if "`xb'`residuals'`shocks'`hdecomp'" == "" {
        local xb  xb
        di as txt "(option xb assumed; fitted values)"
    }

    if "`equation'" != "" {
        if `: word count `equation''>1 {
            disp as error `"Option 'equation' may only contain one token."'
            exit 198
        }
    }
    else {
        local equation "#1"
    }
    
    Depname depname : `equation'
    if "`depname'" == "" {
        di as error "`equation' is not a valid equation name."
        exit 198
    }   

    tempname b
    capture confirm matrix e(b_vargls)  // does not exist for -glsiter(0)-
    if _rc {
        mat `b' = e(b_var)
    }
    else {
        mat `b' = e(b_vargls)
    }

    if "`xb'`residuals'" != "" {
        tempvar xbtemp
        qui mat score double `xbtemp' = `b' if `touse' , equation(`depname')
        if "`xb'"!="" {
            gen `typlist' `varlist' = `xbtemp' if `touse'
            label variable `varlist' "svarih bac: fitted values, equation '`depname''"
        }
        else {
            gen `typlist' `varlist' = `depname' - `xbtemp' if `touse'
            label variable `varlist' "svarih bac: residual, equation '`depname''"
        }
        exit
    }

    local rgmvar     `e(rgmvar)'
    
    capture confirm variable `rgmvar'
    if _rc {
        disp as error `"regime variable `rgmvar' not found"'
        exit 111
    }
    markout `touse' `rgmvar'
    // residuals can be predicted even if the regimevar has missings
    // this is not true for shocks and for historical decompositions, so I mark out these obs

    qui levelsof `rgmvar' if `touse' , local(regimes)  // `regimes' can contain regimes that are not in e(sample)
                                                          // this is not true for llu and bfa

    tempname A B E Dt rowvec rgmmat
    matrix `A' = e(A)
    matrix `B' = e(B)
    matrix `E' = e(E)
    matrix `rgmmat' = e(rgmmat)

    if "`shocks'" != "" {  // calc shock series: (B+E*D_t)^(-1)*A*u_t = e_t

        local numendog = rowsof(`A')
        forvalues i=1/`numendog' {   // need all residuals to compute one shock
            tempname u`i'
            qui svarih_bac_p100 double `u`i'' if `touse', residuals eq(#`i')
            local cnames `cnames' `u`i''
        }

        tempname regimecode BEtA
        tempvar tempshock
        qui gen `typlist' `varlist' = .
        foreach rgm of local regimes {
            capture matrix drop `Dt'
            forvalues i=1/`=rowsof(`rgmmat')' {
                local regimematcode = `rgmmat'[`i',1]
                if `regimematcode'==`rgm' matrix `Dt' = diag(`rgmmat'[`i',2...])
            }
            matrix `BEtA' = inv(`B'+`E'*`Dt') * `A'
            matrix `rowvec' = `BEtA'["`depname'",1...]
            matrix colnames `rowvec' = `cnames'

            capture drop `tempshock'
            qui mat score `typlist' `tempshock' = `rowvec' if `touse' & `rgmvar'==`rgm'
            qui replace `varlist' = `tempshock'            if `touse' & `rgmvar'==`rgm'
        }
        label variable `varlist' "svarih bac: shock, equation '`depname''"
    }

    if "`hdecomp'"!="" {
        
        local stub `varlist'

        local depvar `e(depvar)'
        local exog   `e(exog)'
        local mlag   `e(mlag)'
        
        if "`hdbeg'"=="" | "`hdend'"=="" {
            disp as error `"Option 'hdecomp' must be accompanied by options 'hdbeg' and 'hdend'."'
            exit 198
        }
        
        if !inlist("`unit1'","y","h","q","m","d","b") {        // TODO: make it work for generic and intraday-freqs
            disp as error `"calculation of historical decompositions"
            disp as error `"are implemented only for frequencies y, h, q, m, d, b."'  // some calcs below require integer values of `tvar'
            exit 459
        }
        foreach curopt in hdbeg hdend {
            capture confirm number ``curopt''
            if _rc {
                if "`unit1'"=="b" {
                    capture local num`curopt' = bofd(substr("`tsfmt'",4,.),td(``curopt''))
                }
                else {
                    capture local num`curopt' = t`unit1'(``curopt'')
                }
                if _rc {
                    disp as error `"Could not process date argument to option '`curopt''."'
                    exit 198
                }
            }
            else {
                local num`curopt' ``curopt''
            }
            
            qui count if `tvar'==`num`curopt''
            if `r(N)'!=1 {
                disp as error `"Time value `num`curopt'' from option '`curopt'' does not or not uniquely identify an observation."'
                exit 459
            }
        }        
        
        if `numhdbeg'>`numhdend' {
            disp as error `"Beginning date for historical decomposition after ending date."'
            exit 198
        }
        
        // check that periods that are necessary to compute the step 1 forecast have all nonmissings
        //   also check the exog vars which must have non-missings over the entire prediction period, otherwise missings would be predicted from the occurrence of an exogvar missing on
        tempvar rm rm2
        qui egen int `rm' = rowmiss(`depvar') if inrange(`tvar', `=`numhdbeg'-`mlag'', `=`numhdbeg'-1')
        capture assert `rm' == 0              if inrange(`tvar', `=`numhdbeg'-`mlag'', `=`numhdbeg'-1')
        if _rc {
            disp as error `"Lags of endogenous variables necessary to compute the step 1 forecast contain missing values."'
            exit 416
        }
        tsrevar `exog' , substitute
        local exogtmp `r(varlist)'
        qui egen `rm2' = rowmiss(`exogtmp') if inrange(`tvar',`numhdbeg',`numhdend')
        capture assert `rm2' == 0           if inrange(`tvar',`numhdbeg',`numhdend')
        if _rc {
            disp as error `"Exogenous variables contain missings in forecast span."'
            exit 416
        }
        
        if "`hdshock'"!="" {  // if empty, generate baseline forecast

            qui su `tvar' if e(sample), meanonly
            local smplbeg `r(min)'
            local smplend `r(max)'
            if `numhdbeg'<`smplbeg' | `numhdend'>`smplend' {
                disp as error `"Arguments to options 'hdbeg' and/or 'hdend' not in sample range"'
                exit 198
            }

            if `: word count `hdshock''>1 {
                disp as error `"Option 'hdshock' may only contain one token."'
                exit 198
            }
            Depname shockeq : `hdshock'
            if "`shockeq'" == "" {
                di as error "`hdshock' is not a valid equation name."
                exit 198
            }   


            // check for non-existence of output variables (the forecast variables) and proper naming of variables
            foreach curendog of local depvar {
                capture confirm name `stub'`curendog'
                if _rc {
                    disp as error `"stub `stub' invalid: `stub'`curendog' is not a proper variable name."'
                    exit 198
                }
                capture confirm new variable `stub'`curendog'
                if _rc {
                    disp as error `"Variable `stub'`curendog' already defined."'
                    exit 110
                }
            }

            tempname shock_i
            qui predict `shock_i' if `touse', eq(`shockeq') shocks

            // gen components (i.e. a vector) of red-form resid u, `rp_`curendog''
            //     where "rp": residual part
            //     gen neqs respart variables, each calc as elem (eqnum,i) of A^(-1)*(B+E*D_T) multiplied by `shock_i'
            //     hence I only need col i of A^(-1)*(B+E*D_T), however, this element changes for each regime s so I have to loop over it
            tempname Ainv jnk AinvBEt
            capture matrix `Ainv' = inv(e(A))  // inverting preserves row/colnames
            if _rc {
                disp as error `"matrix e(A) cannot be inverted"'
                exit 498  // return code used by _svard2 for this problem
            }

            tempname temprp
            foreach curendog of local depvar {
                tempname rp_`curendog'   // have to build up tempnames this way in case model variables are long (~max of 32 chars)
                qui gen double `rp_`curendog'' = .
                foreach rgm of local regimes {
                    capture matrix drop `Dt'
                    forvalues i=1/`=rowsof(`rgmmat')' {
                        local regimematcode = `rgmmat'[`i',1]
                        if `regimematcode'==`rgm' matrix `Dt' = diag(`rgmmat'[`i',2...])
                    }
                    matrix `AinvBEt' = `Ainv' * (`B'+`E'*`Dt')
                    matrix `jnk' = `AinvBEt'["`curendog'","`shockeq'"]
                    scalar shock_mlt = `jnk'[1,1]

                    capture drop `temprp'
                    qui gen double `temprp' = `shock_i' * shock_mlt  if `touse' & `rgmvar'==`rgm'
                    qui replace `rp_`curendog'' = `temprp'         if `touse' & `rgmvar'==`rgm'
                }

            }
        }

        foreach curendog of local depvar {
            qui gen double `stub'`curendog' = .
            qui replace `stub'`curendog' = `curendog' if inrange(`tvar', `=`numhdbeg'-`mlag'', `=`numhdbeg'-1')  // note that tvar units must be integers!
        }
        
        // replace column names in b: names of endog vars are now the names of simulated vars
        local bnames : colfullnames `b'
        
        foreach curendog of local depvar {
            local bnames : subinstr local bnames ".`curendog' " ".`stub'`curendog' ", all  // note the dot and the space
        }
        
        matrix colnames `b' = `bnames'
        
        // snippet copied & modified from offical Stata's _varsim.ado
        local j 1
        foreach curendog of local depvar {
            local curendog_new `stub'`curendog'
            local eqj : word `j' of `depvar'
            local scr`j' "score  `curendog_new' = `b', eq(#`j') "
            if "`hdshock'"!="" local upd`j' "update `curendog_new' = `curendog_new' + `rp_`curendog''"  // this line distinguishes baseline from shock_i forecasts
            local j = `j' + 1
        }

        _byobs {
            `scr1'
            `upd1'
            `scr2'
            `upd2'
            `scr3'
            `upd3'
            `scr4'
            `upd4'
            `scr5'
            `upd5'
            `scr6'
            `upd6'
            `scr7'
            `upd7'
            `scr8'
            `upd8'
            `scr9'
            `upd9'
            `scr10'
            `upd10'
            `scr11'
            `upd11'
            `scr12'
            `upd12'
            `scr13'
            `upd13'
            `scr14'
            `upd14'
            `scr15'
            `upd15'
            `scr16'
            `upd16'
            `scr17'
            `upd17'
            `scr18'
            `upd18'
            `scr19'
            `upd19'
            `scr20'
            `upd20'
        } if inrange(`tvar',`numhdbeg',`numhdend')
    
        foreach curendog of local depvar {  // final time series start at hdbeg-1, as in -fcast compute-
            qui replace `stub'`curendog' = . if `tvar'<`=`numhdbeg'-1'
        }

    }

end

*** --------------------------------- SUBROUTINES -----------------------------------------

// modified official Stata routine: returns equation (=variable) name 
program define Depname

    args    depname  /// macro to hold dependent variable name
            colon    /// ":"
            eqopt    //  equation name or #number

    if substr("`eqopt'",1,1) == "#" {
        local eqnum =  substr("`eqopt'", 2,.)
        local dep : word `eqnum' of `e(depvar)'
        c_local `depname' `dep'
        exit
    }
        
    local eqlist  "`e(depvar)'"  // distinction b/w eqlist and deplist only matters when depvar has ts ops, which -svar- allows for ; -svarih- does not
    local deplist "`e(depvar)'"
    local i 1
    while "`dept'" == "" & "`eqlist'" != "" {
        gettoken eqn eqlist : eqlist
        if "`eqn'" == "`eqopt'" {
            local dept : word `i' of `deplist'
            c_local `depname' `dept'
        }
        local i = `i' + 1
    }

end




